------------------------------------------------------------------------------
--
-- UPDATE SCRIPT
--
-- THIS SQL SCRIPT UPDATES DEVICE COUNTER EXTENSION RELEASED IN 8.1 FP4 TO FP15
--
-- NOTE: THIS SHOULD ONLY BE APPLIED TO A COLLECTION OBJECTSERVER
--
------------------------------------------------------------------------------

-----------------------------------------------------------------------
-- Replace the existing device_counter_process_table automation
-----------------------------------------------------------------------
-- The original version of this automation would attempt to process all
-- rows in the master.devices table and build a single string of nvp values
-- for each of the rows in the table.
--
-- If the table contained hundreds of rows then the resulting nvp string
-- was too long to be stored in the 4096 byte ExtendedAttr field, resulting
-- in possible corruption errors logged in the ObjectServer log file when
-- the string was examined at the Aggregation layer.
--
-- The new version of this automation will insert a synthetic event when
-- the string grows to 4000 or more characters long and then start again
-- with an empty nvp string.
--
-- The automation that processes these events at the collection layer
-- is not affected by this and will process multiple synthetic events with
-- no loss of data or functionality.
-----------------------------------------------------------------------

CREATE OR REPLACE PROCEDURE device_counter_insert_synthetic_event(
	in name_value_pairs Char(4096),
	in counter Integer,
	in synthetic_event_counter Integer)
begin
insert into alerts.status (
	Identifier,
	Node,
	Summary,
	Class,
	Type,
	Severity,
	FirstOccurrence,
	LastOccurrence,
	Grade,
	ExpireTime,
	AlertGroup,
	OwnerUID,
	Manager,
	Agent,
	ServerName,
	ExtendedAttr)
values (
	'device_counter_event_' + to_char(getdate()) + ':' + to_char(synthetic_event_counter),
	getservername(),
	'Collection device counter event from ' + getservername(),
	99990,
	13,
	2,
	getdate(),
	getdate(),
	(counter - 1),
	86400,
	'Collection Device Counter',
	65534,
	'OMNIbus Collection Device Counter @' + getservername(),
	'OMNIbus Collection Device Counter',
	getservername(),
	name_value_pairs);
end
go

CREATE OR REPLACE TRIGGER device_counter_process_table
GROUP default_triggers
PRIORITY 20
COMMENT 'Periodically process contents of master.devices table.'
EVERY 1 MINUTES
declare
	name_value_pairs char(4096);
	counter int;
	total_counter int;
	synthetic_event_counter int;
begin

	-- INITIALISE VARIABLE
	set name_value_pairs = '';
	set counter = 0;
	set total_counter = 0;
	set synthetic_event_counter = 0;

	-- PARSE OVER master.devices TABLE AND CONSTRUCT SYNTHETIC
	-- EVENT TO PASS UP TO AGGREGATION LAYER
	for each row device in master.devices
	begin
		-- STORE THE NODE
		set name_value_pairs = nvp_set(
			name_value_pairs,
			to_char(counter),
			device.Node);

		set counter = counter + 1;

		-- STORE THE CLASS OF THE NODE
		set name_value_pairs = nvp_set(
			name_value_pairs,
			to_char(counter),
			device.Class);

		set counter = counter + 1;
		if ( substr( name_value_pairs, 4000, 1 ) <> '' ) then
			-- IN DANGER OF RUNNING OVER THE END OF THE STRING LENGTH
			-- INSERT A SYNTHETIC EVENT TO HOLD THE DEVICE COUNT DETAILS
			execute device_counter_insert_synthetic_event( name_value_pairs, counter, synthetic_event_counter );
			-- KEEP TRACK OF RUNNING TOTAL
			set total_counter = total_counter + counter;
			-- RESET COUNTER OF EVENTS
			set counter=0;
			-- CLEAR NAME VALUE PAIRS STRING
			set name_value_pairs='';
			-- INCREMENT SYNTHETIC EVENT COUNTER
			set synthetic_event_counter = synthetic_event_counter + 1;
		end if;
	end;

	-- INSERT A SYNTHETIC EVENT IF THERE ARE EVENTS REMAINING IN THE BUFFER
	if (counter != 0) then
		-- INSERT A SYNTHETIC EVENT TO HOLD THE DEVICE COUNT DETAILS
		execute device_counter_insert_synthetic_event( name_value_pairs, counter, synthetic_event_counter );
		-- KEEP TRACK OF RUNNING TOTAL
		set total_counter = total_counter + counter;
	end if;
	if( total_counter != 0 ) then
		-- WRITE SUMMARY TO LOG
		write into device_counter_log values (
			to_char(getdate), ': ',
			'Sent ', to_char(total_counter/2),
			' Nodes to the Aggregation layer for device count processing.');
	end if;
	-- CLEAN OUT THE master.devices TABLE
	delete from master.devices;
end;
go
